function [ ] = bench_master( MatSz, VecSz, fname, gstr, cstr )
% bench_master Function which controls the benchmarking of GPUs
%
% For more information see:
%   http://wiki.accelereyes.com/wiki/index.php/Torben%27s_Corner:_Jacket_Benchmark_Tables
%   (C) Torben Larsen, Aalborg University, Denmark. tl@es.aau.dk


%% -------------------------------------------------------------------------------------------------
%  PREPARE BENCHMARKING
%  -------------------------------------------------------------------------------------------------
%  Is DLA present?
try
    inv(grand(3,3));
    dla = '+DLA';
catch
    dla = '-DLA';
end

% Create directory for data if it doesn't exist already
benchdir = strrep(gstr, ' ', '_');
fullbenchdir = ['./benchmarks/' benchdir];
if exist(fullbenchdir)~=7,
    mkdir(fullbenchdir);
end

% Gather GPU info
gpu_info = gpu_entry(13);
gpu_compute = gpu_info.compute;
gpu_system = gpu_info.system;

ginfo_str = evalc('ginfo');
pplace = strfind(ginfo_str,')');
Jver = ginfo_str(1:pplace(1));


%% -------------------------------------------------------------------------------------------------
%  LATEX FILE OPENING
%  -------------------------------------------------------------------------------------------------
% Open data file
Fname = [fullbenchdir '/' fname '.tex'];
fidt = fopen(Fname, 'w');

% Opening statement for Latex document
fprintf(fidt,'\\documentclass[10pt]{article}\n');
fprintf(fidt,'\\usepackage{color}\n');
fprintf(fidt,'\\begin{document}\n\n');

% Make opening statements for a Latex table
fprintf(fidt,'\\begin{table}[htdp]\n');
fprintf(fidt,'\\begin{center}\n');
fprintf(fidt,'\\begin{tabular}{lrrrr} \\hline\n');
fprintf(fidt,'\\multicolumn{5}{c}{{Measured Speed--Up for %s vs. %s}} ', ...
        gstr, strrep(cstr, '-', '--'));
fprintf(fidt,'\\\\  \\hline\n');
fprintf(fidt,'Function$\\qquad$       &  \\multicolumn{2}{c}{Matrix} & ');
fprintf(fidt,'\\multicolumn{2}{c}{Vector}  \\\\ \n'); 
fprintf(fidt,'& $\\quad$ Single$\\quad$ & $\\quad$Double$\\quad$ & ');
fprintf(fidt,'$\\quad$Single$\\quad$ & $\\quad$Double$\\quad$  \\\\ \\hline \n');


%% -------------------------------------------------------------------------------------------------
%  WIKI FILE OPENING
%  -------------------------------------------------------------------------------------------------
% Open data file
Fname = [fullbenchdir '/' fname '.wki'];
fidw = fopen(Fname, 'w');

% Make opening statements for Wiki table
fprintf(fidw,'<center>\n');
fprintf(fidw,'{|border="1" cellpadding="5" cellspacing="0" align="center"\n');
fprintf(fidw,'|+ Table 1: Measured ''''''Jacket'''''' performance as GPU speed-up relative');
fprintf(fidw,' to a CPU. The CPU is running with the maximum number of threads.');
fprintf(fidw,' General matrix size: %5.0f x %5.0f.', MatSz, MatSz);
fprintf(fidw,' General vector size: %5.2fE6 x 1.\n', VecSz/1E6);
fprintf(fidw,' SVD matrix size: %3.0f x %3.0f.', ...
        min(200,floor(MatSz/10)), min(200,floor(MatSz/10)));
fprintf(fidw,' SVD vector size: %3.0f x 1.\n', min(200,floor(MatSz/10)));
fprintf(fidw,' MATLAB version: %s. Jacket version: %s.\n', version, Jver);
fprintf(fidw,' Operating system: %s.\n', gpu_system);

% Define table width etc.
fprintf(fidw,'|-\n');
fprintf(fidw,'! colspan="5" width="550" align="center" style="background:yellow; color:black" |');
fprintf(fidw,' ''''''Jacket'''''' Speed-Up of a %s GPU Relative to a %s CPU\n', gstr, cstr);
fprintf(fidw,'|-\n');
fprintf(fidw,'! rowspan="2" width="150" align="left" style="background:yellow; color:black" | ''''''Function''''''\n');
fprintf(fidw,'! colspan="2" width="200" align="center" style="background:yellow; color:black" | ''''''Matrix''''''\n');
fprintf(fidw,'! colspan="2" width="200" align="center" style="background:yellow; color:black" | ''''''Vector''''''\n');
fprintf(fidw,'|-\n');
fprintf(fidw,'! width="100" align="center" style="background:yellow; color:black" | ''''''Single''''''\n');
fprintf(fidw,'! width="100" align="center" style="background:yellow; color:black" | ''''''Double''''''\n');
fprintf(fidw,'! width="100" align="center" style="background:yellow; color:black" | ''''''Single''''''\n');
fprintf(fidw,'! width="100" align="center" style="background:yellow; color:black" | ''''''Double''''''\n');





%% -------------------------------------------------------------------------------------------------
%  BENCHMARKING
%  -------------------------------------------------------------------------------------------------
%% ALL
Speedup_allMS = all_MS( MatSz );
Speedup_allVS = all_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt all}} & ');
fprintf(fidw,'|-\n| ''''''all''''''\n');
if gpu_compute > 1.2
    Speedup_allMD = all_MD( MatSz );
    Speedup_allVD = all_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_allMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_allMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_allVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_allVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_allMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_allMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_allVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_allVD);

else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_allMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_allVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_allMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_allVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% ANY
Speedup_anyMS = any_MS( MatSz );
Speedup_anyVS = any_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt any}} & ');
fprintf(fidw,'|-\n| ''''''any''''''\n');
if gpu_compute > 1.2
    Speedup_anyMD = any_MD( MatSz );
    Speedup_anyVD = any_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_anyMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_anyMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_anyVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_anyVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_anyMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_anyMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_anyVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_anyVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_anyMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_anyVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_anyMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_anyVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;



%% ASINH
Speedup_asinhMS = asinh_MS( MatSz );
Speedup_asinhVS = asinh_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt asinh}} & ');
fprintf(fidw,'|-\n| ''''''asinh''''''\n');
if gpu_compute > 1.2
    Speedup_asinhMD = asinh_MD( MatSz );
    Speedup_asinhVD = asinh_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_asinhMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_asinhMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_asinhVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_asinhVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_asinhMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_asinhMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_asinhVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_asinhVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_asinhMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_asinhVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_asinhMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_asinhVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% ATAN2
Speedup_atan2MS = atan2_MS( MatSz );
Speedup_atan2VS = atan2_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt atan2}} & ');
fprintf(fidw,'|-\n| ''''''atan2''''''\n');
if gpu_compute > 1.2
    Speedup_atan2MD = atan2_MD( MatSz );
    Speedup_atan2VD = atan2_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_atan2MS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_atan2MD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_atan2VS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_atan2VD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atan2MS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atan2MD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atan2VS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atan2VD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_atan2MS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_atan2VS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atan2MS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atan2VS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% ATAN
Speedup_atanMS = atan_MS( MatSz );
Speedup_atanVS = atan_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt atan}} & ');
fprintf(fidw,'|-\n| ''''''atan''''''\n');
if gpu_compute > 1.2
    Speedup_atanMD = atan_MD( MatSz );
    Speedup_atanVD = atan_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_atanMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_atanMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_atanVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_atanVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atanMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atanMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atanVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atanVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_atanMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_atanVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atanMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_atanVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% BESSELJ
Speedup_besseljMS = besselj_MS( MatSz );
Speedup_besseljVS = besselj_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt besselj}} & ');
fprintf(fidw,'|-\n| ''''''besselj''''''\n');
if gpu_compute > 1.2
    Speedup_besseljMD = besselj_MD( MatSz );
    Speedup_besseljVD = besselj_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_besseljMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_besseljMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_besseljVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_besseljVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_besseljMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_besseljMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_besseljVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_besseljVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_besseljMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_besseljVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_besseljMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_besseljVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% CHOL
Speedup_cholMS = chol_MS( MatSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt chol}} & ');
fprintf(fidw,'|-\n| ''''''chol''''''\n');
if strcmp(dla,'+DLA')
    Speedup_cholMD = chol_MD( MatSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_cholMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_cholMD);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_cholMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_cholMD);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_cholMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_cholMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% CONV2
Speedup_conv2MS = conv2_MS( MatSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt conv2}} & ');
fprintf(fidw,'|-\n| ''''''conv2''''''\n');
if gpu_compute > 1.2
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_conv2MS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_conv2MS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_conv2MS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_conv2MS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% COS
Speedup_cosMS = cos_MS( MatSz );
Speedup_cosVS = cos_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt cos}} & ');
fprintf(fidw,'|-\n| ''''''cos''''''\n');
if gpu_compute > 1.2
    Speedup_cosMD = cos_MD( MatSz );
    Speedup_cosVD = cos_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_cosMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_cosMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_cosVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_cosVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_cosMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_cosMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_cosVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_cosVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_cosMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_cosVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_cosMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_cosVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% DET
Speedup_detMS = det_MS( MatSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt det}} & ');
fprintf(fidw,'|-\n| ''''''det''''''\n');
if strcmp(dla,'+DLA')
    Speedup_detMD = det_MD( MatSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_detMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_detMD);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_detMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_detMD);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_detMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_detMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% EXP
Speedup_expMS = exp_MS( MatSz );
Speedup_expVS = exp_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt exp}} & ');
fprintf(fidw,'|-\n| ''''''exp''''''\n');
if gpu_compute > 1.2
    Speedup_expMD = exp_MD( MatSz );
    Speedup_expVD = exp_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_expMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_expMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_expVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_expVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_expMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_expMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_expVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_expVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_expMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_expVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_expMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_expVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% FFT
%Speedup_fftMS = fft_MS( 2048 );
%Speedup_fftVS = fft_VS( 2^22 );
%fprintf(fidt,'\\textcolor{blue}{{\\tt fft}} & ');
%fprintf(fidw,'|-\n| ''''''fft''''''\n');
%if gpu_compute > 1.2
%    Speedup_fftMD = fft_MD( MatSz );
%    Speedup_fftVD = fft_VD( VecSz );
%    
%    % Latex
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_fftMS);
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_fftMD);
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_fftVS);
%    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_fftVD);
%    
%    % Wiki
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_fftMS);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_fftMD);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_fftVS);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_fftVD);
%else
%    % Latex
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_fftMS);
%    fprintf(fidt,'---$\\quad\\;$ & ');
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_fftVS);
%    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
%
%    % Wiki
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_fftMS);
%    fprintf(fidw,'! align="right" | ---\n');
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_fftVS);
%    fprintf(fidw,'! align="right" | ---\n');
%end
%gcache flush;


%% FIND
Speedup_findMS = find_MS( MatSz );
Speedup_findVS = find_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt find}} & ');
fprintf(fidw,'|-\n| ''''''find''''''\n');
if gpu_compute > 1.2
    Speedup_findMD = find_MD( MatSz );
    Speedup_findVD = find_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_findMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_findMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_findVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_findVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_findMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_findMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_findVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_findVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_findMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_findVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_findMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_findVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% IFFT
%Speedup_ifftMS = ifft_MS( 2048 );
%Speedup_ifftVS = ifft_VS( 2^22 );
%fprintf(fidt,'\\textcolor{blue}{{\\tt ifft}} & ');
%fprintf(fidw,'|-\n| ''''''ifft''''''\n');
%if gpu_compute > 1.2
%    Speedup_ifftMD = ifft_MD( MatSz );
%    Speedup_ifftVD = ifft_VD( VecSz );
%    
%    % Latex
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_ifftMS);
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_ifftMD);
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_ifftVS);
%    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_ifftVD);
%    
%    % Wiki
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_ifftMS);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_ifftMD);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_ifftVS);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_ifftVD);
%else
%    % Latex
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_ifftMS);
%    fprintf(fidt,'---$\\quad\\;$ & ');
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_ifftVS);
%    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
%
%    % Wiki
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_ifftMS);
%    fprintf(fidw,'! align="right" | ---\n');
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_ifftVS);
%    fprintf(fidw,'! align="right" | ---\n');
%end
%gcache flush;


%% INTERP2
Speedup_interp2MS = interp2_MS( MatSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt interp2}} & ');
fprintf(fidw,'|-\n| ''''''interp2''''''\n');
if gpu_compute > 1.2
    Speedup_interp2MD = interp2_MD( MatSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_interp2MS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_interp2MD);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_interp2MS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_interp2MD);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_interp2MS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_interp2MS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% INV

fprintf(fidt,'\\textcolor{blue}{{\\tt inv}} & ');
fprintf(fidw,'|-\n| ''''''inv''''''\n');
if strcmp(dla,'+DLA')
    Speedup_invMD = inv_MD( MatSz );
    
    % Latex
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_invMD);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
    
    % Wiki
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_invMD);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
else
    % Latex
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% LOG
Speedup_logMS = log_MS( MatSz );
Speedup_logVS = log_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt log}} & ');
fprintf(fidw,'|-\n| ''''''log''''''\n');
if gpu_compute > 1.2
    Speedup_logMD = log_MD( MatSz );
    Speedup_logVD = log_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_logMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_logMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_logVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_logVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_logMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_logMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_logVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_logVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_logMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_logVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_logMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_logVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% LU
Speedup_luMS = lu_MS( MatSz );
Speedup_luVS = lu_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt lu}} & ');
fprintf(fidw,'|-\n| ''''''lu''''''\n');
if strcmp(dla,'+DLA')
    Speedup_luMD = lu_MD( MatSz );
    Speedup_luVD = lu_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_luMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_luMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_luVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_luVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_luMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_luMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_luVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_luVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_luMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_luVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_luMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_luVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% MAX
Speedup_maxMS = max_MS( MatSz );
Speedup_maxVS = max_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt max}} & ');
fprintf(fidw,'|-\n| ''''''max''''''\n');
if gpu_compute > 1.2
    Speedup_maxMD = max_MD( MatSz );
    Speedup_maxVD = max_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_maxMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_maxMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_maxVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_maxVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_maxMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_maxMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_maxVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_maxVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_maxMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_maxVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_maxMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_maxVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% MIN
Speedup_minMS = min_MS( MatSz );
Speedup_minVS = min_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt min}} & ');
fprintf(fidw,'|-\n| ''''''min''''''\n');
if gpu_compute > 1.2
    Speedup_minMD = min_MD( MatSz );
    Speedup_minVD = min_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_minMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_minMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_minVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_minVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_minMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_minVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% MINUS
Speedup_minusMS = minus_MS( MatSz );
Speedup_minusVS = minus_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt minus}} & ');
fprintf(fidw,'|-\n| ''''''minus''''''\n');
if gpu_compute > 1.2
    Speedup_minusMD = minus_MD( MatSz );
    Speedup_minusVD = minus_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_minusMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_minusMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_minusVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_minusVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minusMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minusMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minusVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minusVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_minusMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_minusVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minusMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_minusVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% MLDIVIDE
Speedup_mldivideMS = mldivide_MS( MatSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt mldivide}} & ');
fprintf(fidw,'|-\n| ''''''mldivide''''''\n');
if strcmp(dla,'+DLA')
    Speedup_mldivideMD = mldivide_MD( MatSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_mldivideMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_mldivideMD);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_mldivideMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_mldivideMD);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_mldivideMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_mldivideMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% MPOWER
Speedup_mpowerMS = mpower_MS( MatSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt mpower}} & ');
fprintf(fidw,'|-\n| ''''''mpower''''''\n');
if strcmp(dla,'+DLA')
    Speedup_mpowerMD = mpower_MD( MatSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_mpowerMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_mpowerMD);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_mpowerMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_mpowerMD);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_mpowerMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_mpowerMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% NORM
Speedup_normMS = norm_MS( MatSz );
Speedup_normVS = norm_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt norm}} & ');
fprintf(fidw,'|-\n| ''''''norm''''''\n');
if strcmp(dla,'+DLA')
    Speedup_normMD = norm_MD( MatSz );
    Speedup_normVD = norm_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_normMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_normMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_normVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_normVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_normMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_normMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_normVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_normVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_normMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_normVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_normMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_normVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% PLUS
Speedup_plusMS = plus_MS( MatSz );
Speedup_plusVS = plus_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt plus}} & ');
fprintf(fidw,'|-\n| ''''''plus''''''\n');
if gpu_compute > 1.2
    Speedup_plusMD = plus_MD( MatSz );
    Speedup_plusVD = plus_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_plusMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_plusMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_plusVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_plusVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_plusMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_plusMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_plusVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_plusVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_plusMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_plusVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_plusMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_plusVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% POWER
Speedup_powerMS = power_MS( MatSz );
Speedup_powerVS = power_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt power}} & ');
fprintf(fidw,'|-\n| ''''''power''''''\n');
if gpu_compute > 1.2
    Speedup_powerMD = power_MD( MatSz );
    Speedup_powerVD = power_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_powerMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_powerMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_powerVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_powerVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_powerMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_powerMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_powerVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_powerVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_powerMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_powerVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_powerMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_powerVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% RAND
Speedup_randMS = rand_MS( MatSz );
Speedup_randVS = rand_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt rand}} & ');
fprintf(fidw,'|-\n| ''''''rand''''''\n');
if gpu_compute > 1.2
    Speedup_randMD = rand_MD( MatSz );
    Speedup_randVD = rand_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_randMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_randMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_randVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_randVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_randMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_randVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% RANDN
Speedup_randnMS = randn_MS( MatSz );
Speedup_randnVS = randn_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt randn}} & ');
fprintf(fidw,'|-\n| ''''''randn''''''\n');
if gpu_compute > 1.2
    Speedup_randnMD = randn_MD( MatSz );
    Speedup_randnVD = randn_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_randnMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_randnMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_randnVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_randnVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randnMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randnMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randnVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randnVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_randnMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_randnVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randnMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_randnVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;



%% RDIVIDE
Speedup_rdivideMS = rdivide_MS( MatSz );
Speedup_rdivideVS = rdivide_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt rdivide}} & ');
fprintf(fidw,'|-\n| ''''''rdivide''''''\n');
if gpu_compute > 1.2
    Speedup_rdivideMD = rdivide_MD( MatSz );
    Speedup_rdivideVD = rdivide_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_rdivideMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_rdivideMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_rdivideVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_rdivideVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_rdivideMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_rdivideMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_rdivideVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_rdivideVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_rdivideMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_rdivideVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_rdivideMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_rdivideVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% SORT
%Speedup_sortMS = sort_MS( MatSz );
%Speedup_sortVS = sort_VS( VecSz );
%fprintf(fidt,'\\textcolor{blue}{{\\tt sort}} & ');
%fprintf(fidw,'|-\n| ''''''sort''''''\n');
%if gpu_compute > 1.2
%    Speedup_sortMD = sort_MD( MatSz );
%    Speedup_sortVD = sort_VD( VecSz );
%    
%    % Latex
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_sortMS);
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_sortMD);
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_sortVS);
%    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_sortVD);
%    
%    % Wiki
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sortMS);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sortMD);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sortVS);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sortVD);
%else
%    % Latex
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_sortMS);
%    fprintf(fidt,'---$\\quad\\;$ & ');
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_sortVS);
%    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
%
%    % Wiki
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sortMS);
%    fprintf(fidw,'! align="right" | ---\n');
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sortVS);
%    fprintf(fidw,'! align="right" | ---\n');
%end
%gcache flush;


%% SUBSASGN
Speedup_subsasgnMS = subsasgn_MS( MatSz );
Speedup_subsasgnVS = subsasgn_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt subsasgn}} & ');
fprintf(fidw,'|-\n| ''''''subsasgn''''''\n');
if gpu_compute > 1.2
    Speedup_subsasgnMD = subsasgn_MD( MatSz );
    Speedup_subsasgnVD = subsasgn_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_subsasgnMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_subsasgnMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_subsasgnVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_subsasgnVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_subsasgnMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_subsasgnMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_subsasgnVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_subsasgnVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_subsasgnMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_subsasgnVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_subsasgnMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_subsasgnVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% SUM
Speedup_sumMS = sum_MS( MatSz );
Speedup_sumVS = sum_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt sum}} & ');
fprintf(fidw,'|-\n| ''''''sum''''''\n');
if gpu_compute > 1.2
    Speedup_sumMD = sum_MD( MatSz );
    Speedup_sumVD = sum_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_sumMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_sumMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_sumVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_sumVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sumMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sumMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sumVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sumVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_sumMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_sumVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sumMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_sumVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% SVD
Speedup_svdMS = svd_MS( min(200,floor(MatSz/10)) );
Speedup_svdVS = svd_VS( min(200,floor(MatSz/10)) );
fprintf(fidt,'\\textcolor{blue}{{\\tt svd}} & ');
fprintf(fidw,'|-\n| ''''''svd''''''\n');
if gpu_compute > 1.2
    Speedup_svdMD = svd_MD( min(200,floor(MatSz/10)) );
    Speedup_svdVD = svd_VD( min(200,floor(MatSz/10)) );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_svdMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_svdMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_svdVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_svdVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_svdMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_svdMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_svdVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_svdVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_svdMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_svdVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_svdMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_svdVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% TIMES
Speedup_timesMS = times_MS( MatSz );
Speedup_timesVS = times_VS( VecSz );
fprintf(fidt,'\\textcolor{blue}{{\\tt times}} & ');
fprintf(fidw,'|-\n| ''''''times''''''\n');
if gpu_compute > 1.2
    Speedup_timesMD = times_MD( MatSz );
    Speedup_timesVD = times_VD( VecSz );
    
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_timesMS);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_timesMD);
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_timesVS);
    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_timesVD);
    
    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_timesMS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_timesMD);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_timesVS);
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_timesVD);
else
    % Latex
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_timesMS);
    fprintf(fidt,'---$\\quad\\;$ & ');
    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_timesVS);
    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');

    % Wiki
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_timesMS);
    fprintf(fidw,'! align="right" | ---\n');
    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_timesVS);
    fprintf(fidw,'! align="right" | ---\n');
end
gcache flush;


%% TRAPZ
%Speedup_trapzMS = trapz_MS( MatSz );
%Speedup_trapzVS = trapz_VS( VecSz );
%fprintf(fidt,'\\textcolor{blue}{{\\tt trapz}} & ');
%fprintf(fidw,'|-\n| ''''''trapz''''''\n');
%if gpu_compute > 1.2
%    Speedup_trapzMD = trapz_MD( MatSz );
%    Speedup_trapzVD = trapz_VD( VecSz );
%    
%    % Latex
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_trapzMS);
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_trapzMD);
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_trapzVS);
%    fprintf(fidt,'$%7.2f\\quad\\;$ \\\\ \n', Speedup_trapzVD);
%    
%    % Wiki
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_trapzMS);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_trapzMD);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_trapzVS);
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_trapzVD);
%else
%    % Latex
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_trapzMS);
%    fprintf(fidt,'---$\\quad\\;$ & ');
%    fprintf(fidt,'$%7.2f\\quad\\;$ & ', Speedup_trapzVS);
%    fprintf(fidt,'---$\\quad\\;$ \\\\ \n');
%
%    % Wiki
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_trapzMS);
%    fprintf(fidw,'! align="right" | ---\n');
%    fprintf(fidw,'! align="right" | %7.2f\n', Speedup_trapzVS);
%    fprintf(fidw,'! align="right" | ---\n');
%end
%gcache flush;





%% -------------------------------------------------------------------------------------------------
%  MAKE END STATEMENTS OF LATEX TABLE
%  -------------------------------------------------------------------------------------------------
fprintf(fidt,'\\hline \n\\end{tabular} \n');
fprintf(fidt,'\\end{center} \n');
fprintf(fidt,'\\caption{Measured {\\tt Jacket} performance as GPU speed-up relative\n');
fprintf(fidt,' to a CPU. The CPU is running with the maximum number of threads.\n');
fprintf(fidt,' Matrix size: $%5.0f \\times %5.0f$.', MatSz, MatSz);
fprintf(fidt,' Vector size: $%5.2f \\cdot 10^{6} \\times 1$.\n', VecSz/1E6);
fprintf(fidt,' {\\tt svd} matrix size: $%3.0f \\times %3.0f$.', ...
	    min(200,floor(MatSz/10)), min(200,floor(MatSz/10)));
fprintf(fidt,' {\\tt svd} vector size: $%3.0f \\times 1$.\n', min(200,floor(MatSz/10)));
fprintf(fidt,' MATLAB version: %s. Jacket version: %s.\n', version, Jver);
fprintf(fidt,' Operating system: %s.}\n', gpu_system);
fprintf(fidt,'\\label{tab:XXX}\n');
fprintf(fidt,'\\end{table}\n\n');
fprintf(fidt,'\\end{document}\n');

% Close file
fclose(fidt);



%% -------------------------------------------------------------------------------------------------
%  MAKE END STATEMENTS OF WIKI TABLE
%  -------------------------------------------------------------------------------------------------
fprintf(fidw,'|}\n</center>\n');

% Close file
fclose(fidw);




%% SAVE DATA
Fname = [fullbenchdir '/' fname '.mat'];
save(Fname, '-regexp', 'Speedup_');

end